/*
  tty routines
  Copyright (C) 1991 by Network Wizards
*/

/*
  tty routines:
    tty_init_data()	           initialize random data structures
    tty_init_int()                 initialize/install tty interrupt handlers
    tty_init_uart()	           initialize uarts (and test existence)
    send_break()                   send a break signal
    tty_tx(char)                   send a character
    tty_txs(string)                send a string
    tty_rx(char)		   recv a character
*/

#include <stdio.h>
#include <dos.h>
#include <alloc.h>
#include <time.h>
#include "okicon.h"
#include "tty.h"

#define PIC   0x20	/* port of PIC */
#define IMR   0x21	/* Interrupt Mask Register in PIC */
#define EOI   0x20	/* end of interrupt command to PIC */

/* uart register offsets */
#define RBR 0		/* R:  receive regiser (DLAB = 0) */
#define THR 0		/* W:  transmit hold register (DLAB = 0) */
#define DLL 0		/* RW: baud divisor low byte (DLAB = 1)  */
#define IER 1		/* RW: interrupt enable register (DLAB = 0) */
#define DLM 1		/* RW: baud divisor high byte (DLAB = 1)  */
#define IIR 2		/* R:  interrupt ident register */
#define LCR 3		/* RW: line control register */
#define MCR 4		/* RW: modem control register */
#define LSR 5		/* R:  line status register */
#define MSR 6		/* R:  modem status register */
#define SCR 7		/* RW: scratch register */

/* uart register bits */
#define DLAB 0x80	/* divisor latch access bit (LCR) */

/* interrupt enable register */
#define ERBFI 0x01      /* enable received data int */
#define ETBEI 0x02      /* enable transmitter holder register empty int */
#define ELSI  0x04      /* enable receiver line status int */
#define EDSSI 0x08      /* enable modem status int */

/* LCR bits */
#define BRK_TIME  250   /* BREAK for this many milliseconds */
#define BRK  0x40	/* send a BREAK */

/* Modem Control Register (MCR) */
#define DTR  0x01
#define RTS  0x02
#define LOOP 0x10	/* set loopback mode */

/* Line Status Register (LSR) */
#define DR   0x01	/* data received */
#define OE   0x02	/* overrun error */
#define PE   0x04	/* parity error */
#define FE   0x08	/* framing error */

/* Modem Status Register (MSR) */
#define CTS  0x10
#define DSR  0x20
#define RI   0x40
#define DCD  0x80

struct intrs
{
  void interrupt (*func)();
} intrs[16];

struct bpsstruct bps_tab[] =
{
  "50",    50,    0x09, 0x00,
  "110",   110,   0x04, 0x17,
  "300",   300,   0x01, 0x80,
  "600",   600,   0x00, 0xC0,
  "1200",  1200,  0x00, 0x60,
  "2400",  2400,  0x00, 0x30,
  "4800",  4800,  0x00, 0x18,
  "7200",  7200,  0x00, 0x10,
  "9600",  9600,  0x00, 0x0c,	   
  "14400", 14400, 0x00, 0x09,
  "19200", 19200, 0x00, 0x06,
};
int n_bps = (sizeof(bps_tab)/sizeof(struct bpsstruct));

/* forward declarations */
void init_interfaces();
void tty_init_uart();
void set_uart_params();
void ignore_line();
void tty_txs();
void tty_tx();
tty_rx();
void send_break();
void tty_init_int();
void init_uart_intrs();
void tty_undo_int();

extern speed;

byte beepreq;			/* number of intervals to beep for */
byte old_imr;

/* serial line stuff */
word f_uart;			/* address of uart */
byte f_irq;			/* interrupt request line */
bool f_txing;			/* true if transmitting a char */
char *f_outbufa;		/* line output buffer address */
word f_outbufptr;		/* ptr to next char to tx */
word f_outbuflast;		/* ptr to next free buffer position */
word f_outbufsz;		/* total size of buffer */
word f_outbufcnt;		/* number of chars in buffer */

word f_lsr_oe;			/* overrun errors */
word f_lsr_pe;			/* parity errors */
word f_lsr_fe;			/* framing errors */
bool f_dsrmon;			/* true if want to monitor dsr */
byte f_msr;			/* modem status register */

/* flow control shit */
bool f_txoff;			/* true if we sent xoff */
bool f_txp;			/* true if allowed to tx */
bool f_fcsoftin;		/* true if flow-soft-in */
bool f_fcsoftout;		/* true if flow-soft-out */
bool f_fchardin;		/* true if flow-hard-in	*/
bool f_fchardout;		/* true if flow-hard-in	*/
bool f_rqsxoff;			/* true if need to send XOFF */

/*
  bigbuf holds input characters from uarts awaiting processing.
*/
char *bb_buf;				/* pointer to bigbuf */
word bb_last;				/* pointer to location after bigbuf */
word bb_read;				/* index to last entry read */
word bb_write;				/* index to next entry to write */
word bb_count;				/* number of buffer entries */
word bb_hi_limit;			/* size at which to flow-control */
word bb_lo_limit;			/* size at which to unflow-control */
word bb_ovflow;				/* count of overflows */

/* initialize tty data */
void init_tty_data()
{
  char *p;
  word size;
  
  f_fcsoftin = TRUE;
  f_fcsoftout = TRUE;
  f_fchardin = FALSE;
  f_fchardout = FALSE;

/* transmit buffer */
  f_outbufptr = 0;
  f_outbuflast = 0;
  f_outbufcnt = 0;
  f_outbufsz = OUTBUF_DEFSZ;
  if ((p = (char *) malloc(f_outbufsz)) == NULL)
  {
    fprintf(stderr,"?Not enough memory for output buffer\n");
    exit(1);
  }
  f_outbufa = p;

  f_msr =     0;
  f_txing =   FALSE;
  f_txp =     TRUE;
  f_txoff =   FALSE;
  f_rqsxoff = FALSE;

  f_lsr_oe = 0;
  f_lsr_pe = 0;
  f_lsr_fe = 0;

/* receive buffer */
  size = 2048;
  if ((bb_buf = (char *) malloc(size)) == NULL)
  {
    fprintf(stderr,"?Not enough memory for input-buffer\n");
    exit(1);
  }
  bb_last = (word) bb_buf + size;
  bb_read = (word) bb_buf;
  bb_write = (word) bb_buf + 1;
  bb_count = 0;
  bb_hi_limit = size - 128;
  bb_lo_limit = 128;
  bb_ovflow = 0;
}

/*
  for a given line, check to see if uart exists,
  and intialize if so.  turn uart interrupts off for now.
*/
void tty_init_uart()
{
  int i, uart;
  byte scratch;

  uart = f_uart;

  /* test for working uart by reading a register */
  if (inportb(uart+IIR) & 0xf8)
  {
    fprintf(stderr,"%%UART for line not responding.\n");
    exit(1);
  }

  /* disable uart interrupts */
  outportb(uart+IER,0);
  delay(1);

  set_uart_params();     /* set speed, data bits, stop bits, parity */

  /* run a quick self-test */
  outportb(uart+MCR,LOOP);		/* set loopback mode */
  delay(1);

  while (inportb(uart+IIR) != 1)	/* empty out any garbage */
  {
    inportb(uart+LSR);
    inportb(uart+RBR);
    inportb(uart+MSR);
  }

  /* test modem control signals */
  if ((inportb(uart+MSR) & (CTS|DSR|RI|DCD)) != 0)
  {
    ignore_line();
    return;
  }
  outportb(uart+MCR,LOOP|DTR|RTS|OUT1|OUT2);
  delay(1);
  if ((inportb(uart+MSR) & (CTS|DSR|RI|DCD)) != (CTS|DSR|RI|DCD))
  {
    ignore_line();
    return;
  }
  outportb(uart+MCR,LOOP);		/* set loopback mode */
  delay(1);

  /* test sending and receiving chars */
  for (i = 0; i < 10; i++)
  {
    int j;
    
    outportb(uart+THR,i);		/* transmit a char */
    delay(1);
    for (j = 0; j < 50; j++)
    {
      delay(10);
      if ((inportb(uart+LSR) & (DR|OE|PE|FE)) == DR)
	goto gotchar;
    }
    ignore_line();
    return;
gotchar:
    if (inportb(uart+RBR) != i)
    {
      ignore_line();
      return;
    }
  }
  outportb(uart+MCR,0);			/* turn off loopback */
/********************** temporary hack for testing: turn on DTR and RTS**/
/*
delay(1);
outportb(uart+MCR,(DTR|RTS));
*/
  delay(1);
  while (inportb(uart+IIR) != 1)	/* empty out any garbage */
  {
    delay(1);
    inportb(uart+LSR);
    delay(1);
    inportb(uart+RBR);
    delay(1);
    inportb(uart+MSR);
  }
}

/* set speed, data bits, stop bits and parity for the line  */
void set_uart_params()
{
  int i, uart;
  byte scratch;
  int char_bits, stop_bits, parity;

  uart = f_uart;

  /* setup uart params */
  outportb(uart+LCR,DLAB);		/* access baud rate registers */
  delay(1);
  for (i = 0; i < n_bps; i++)
    if (speed == bps_tab[i].speed)
      break;
  outportb(uart+DLM,bps_tab[i].hvalue);
  delay(1);
  outportb(uart+DLL,bps_tab[i].lvalue);
  delay(1);

  char_bits = 8;
  stop_bits = 1;
  parity =    0;
  
  if (char_bits == 7)
    scratch = 2;
  else
  if (char_bits == 8)
    scratch = 3;
  if (stop_bits == 2)
    scratch |= 4;
  scratch |= parity;
  outportb(uart+LCR,scratch);
  delay(1);
}

void ignore_line()
{
  fprintf(stderr,"%%Line failed internal loopback test.\n");
  exit(1);
}

/* transmit a string */
void tty_txs(s)
  char *s;
{
  while (*s != '\0')
    tty_tx(*s++);
}

/* transmit a character */
void tty_tx(ch)
  byte ch;
{
  if (ttydebugp)
    printf("<%02x>",ch);
  disable();
  if ((f_txing) ||			/* if a char is being transmitted */
      (!f_txp) ||			/* or we're flow-controlled */
      (f_outbufcnt))			/* or output buffer not empty */
  {					/* then buffer it for output later */
    if (f_outbufcnt < f_outbufsz)
    {
      word c;

      c = f_outbuflast++;
      *(f_outbufa + c) = ch;
      if (c >= (f_outbufsz - 1))
        f_outbuflast = 0;
      f_outbufcnt++;
    }
    else
    {
      /* fprintf(stderr,"%%Output buffer overflow on line\n"); */
      beepreq |= 1;
    }
  }
  else      /* nothing buffered or being sent, so we can send immediately */
  {
    outportb(f_uart+THR,ch);
    f_txing = TRUE;
  }
  enable();
}

/* receive a character, return false if none in buffer */
tty_rx(ch)
  byte *ch;
{
  byte status;
  
  if (!bb_count)			/* buffer is empty */
    return(FALSE);
  disable();
  bb_read++;				/* point to next char to read */
  if (bb_read >= bb_last)		/* wrap? */
    bb_read = (word) bb_buf;		/* yes, reset to BOB */
  *ch = *((char *) bb_read);		/* get char from buffer */
  bb_count--;				/* update buffer entry count */
  enable();
  if (bb_count == bb_lo_limit)		/* is buffer emptying out? */
  {
    if (f_fchardin)			/* hardware flow control? */
    {
      status = inportb(f_uart+MCR);
      status |= RTS;			/* turn RTS on */
      outportb(f_uart+MCR,status);
    }
    if (f_txoff)			/* was line stopped by xoff? */
      tty_tx(XON);			/* yes, send XON */
  }
  if (ttydebugp)
    printf("{%02x}",*ch);
  return(TRUE);
}

/* flush input buffer */
tty_rx_flush()
{
  char c;
  
  while (tty_rx(&c));
}

/* send a BREAK signal to line */
void send_break()
{
  byte lcr;
  
  /* wait for transmitter to empty */
  for (;;)
    if (!(f_txing ||		/* if a char is not being transmitted */
          f_outbufcnt))		/* and output buffer is empty */
      break;			/* then we're ready to tx */
  lcr = inportb(f_uart+LCR);  
  lcr |= BRK;
  outportb(f_uart+LCR,lcr);
  delay(BRK_TIME);		/* break for a while */
  lcr &= ~BRK;
  outportb(f_uart+LCR,lcr);
}

/*
 initialize interrupt handlers.
 setup interrupt vectors, and enable uart interrupts.
*/
void tty_init_int()
{
  int i;
  byte imr, irq;
  
  /* setup new interrupt vectors for uarts */
  for (i = 0; i < 16; i++)
    intrs[i].func = NULL;
  imr = inportb(IMR);			/* read current interrupt mask */
  old_imr = imr;			/* save for undoing */

  irq = f_irq;
  intrs[irq].func = getvect(IRQ0+irq);
  setvect(IRQ0+irq,tty_int);
  if ((~old_imr) & (1 << irq))
    printf("%%Possible IRQ %d device conflict.\n",irq);
  imr &= ~ (1 << irq);
  outportb(IMR,imr);			/* enable uart interrupts in PIC */
  init_uart_intrs();
}

void init_uart_intrs()
{
  /* now turn on uart interrupts */
  outportb(f_uart+MCR,(DTR | RTS | OUT2));
  delay(1);
  outportb(f_uart+IER,(ERBFI | ETBEI | ELSI | EDSSI));
  delay(1);
}

/* turn DTR on or off */
dtr(state)
  bool state;
{
  byte status;
  
  status = inportb(f_uart+MCR);
  if (state)
    status |= DTR;
  else
    status &= ~DTR;
  outportb(f_uart+MCR,status);
}

/* put back original uart interrupts */
void tty_undo_int()
{
  byte i, mcr;

  mcr = inportb(f_uart+MCR);
  mcr ^= (OUT1 | OUT2);			/* turn off uart intr line */
  outportb(f_uart+MCR,mcr);

  outportb(IMR,old_imr);		/* reset PIC IMR */
  for (i = 0; i < 16; i++)
    if (intrs[i].func != NULL)
      setvect(IRQ0+i,intrs[i].func);
}
