/*
  OKI Cellular Telephone Control Library
  Copyright (C) 1993, 1994 by Network Wizards
*/

#include <stdio.h>
// #include <mem.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#define FALSE		0
#define TRUE		1

#define VERSION	        "1.4(940618)"   /* major.minor(last-edit-date) */

typedef unsigned char bool;
typedef unsigned char byte;
typedef unsigned int  word;
typedef int bint;

// #include "ttyio.h"
#include "ctlib.h"
#include "ttyio.c"

void ct_rx_flush();

/* Memory locations (bytes) */
/* 900 series */
#define EE_NAMBLOCK_9	0xa02b		/* nam block (by 0x40s) */
#define EE_NAMINC_9     0x40
#define EE_PROGPWD_9	0xbeda		/* dealer pwd (5 bcd) */
#define EE_CUSTMSG_9	0xbeaf		/* custom poweron msg */
#define	EE_NAMINDEX_9   0xbf2c		/* index of selected nam */
#define EE_NAMELEN_9    8
#define RAM_STRM_POS_9	0x27		/* byte holding stream bit */
#define RAM_STRM_BIT_9	5		/* stream selector bit position */
#define RAM_SYSID_9     0xc06a		/* received system id */
/* 1100 series */
#define EE_NAMBLOCK_11	0xff80		/* nam block */
#define EE_NAMINC_11    1
#define EE_PROGPWD_11	0xf083		/* dealer pwd (5 bcd) */
#define EE_CUSTMSG_11	0xf088		/* custom poweron msg */
#define	EE_NAMINDEX_11  0xf020		/* index of selected nam */
#define EE_NAMELEN_11   10
#define RAM_STRM_POS_11	0x104
#define RAM_STRM_BIT_11	1
/* don't know yet */
#define RAM_SYSID_11    0x0		/* received system id */

char TONE_CHARS[]    = " 1234567890*#ABCD";
char TX_TONE_CHARS[] = " 1234567890*#ABCDefghijkl";
char OKI_MEM_DEFS[]  = ".1234567890*#..P";

byte rxbuf[4];

#define BUFSZ	1024
byte inbuf[BUFSZ], outbuf[BUFSZ];

word ee_namblock;
byte ee_naminc;
word ee_progpwd;
word ee_custmsg;
word ee_namindex;
byte ee_namelen;
word ram_strm_pos;
byte ram_strm_bit;
word ram_sysid;

char ct_rom_version[8];
int  ct_model;

int ct_got_namblock;
byte ct_namblock[80];			/* nam info block */
struct nam_data nam_info[5];
unsigned long ct_esn;

struct ct_state_struct ct_state;

/* initialize ct and serial library */
bint ct_lib_init(model,addr,irq)
  word model;
  word addr;
  byte irq;
{
  if ((model != 900) && (model != 1150) && (model != 1200))
    return(FALSE);
  delay(0);
  if (!tty_set_comm_params(addr,irq,"2400",8,'N',1,0))
    return(FALSE);
  tty_set_data(inbuf,BUFSZ,outbuf,BUFSZ,0,FALSE);
  tty_init_int();
  OKI_MEM_DEFS[0] = '\0';
  ct_state.carrier = FALSE;
  ct_state.sigtone = FALSE;
  ct_state.rxaudio = FALSE;
  ct_state.txaudio = FALSE;
  ct_state.compexp = FALSE;
  ct_state.powerlevel = 0;
  ct_state.sat = 0;
  ct_state.volume = 0;
  ct_state.audiopath = AUDIO_EARPIECE;
  ct_state.channel = 333;
  ct_state.namindex = 0;
  ct_state.oldnam = -1;
  return(TRUE);
}

void ct_lib_done()
{
  byte blk[8];

  tty_undo_int();
}

/* power up oki and tell it what we are */
bint ct_on(mode)
  word mode;
{
  word msg;
  byte cmd, data;
  byte blk[8];
    
  ct_esn = 0;
  ct_got_namblock = FALSE;
  ct_rx_flush();
  ct_pic_state(TRUE,TRUE);
  for (;;)
  {
    if (!ct_oki_get_msg(&cmd,&data,3000))
      return(FALSE);
    if (((cmd == 0x90) || (cmd == 0x94)) && (data == 0x01))
    {
      ct_tx(0x0404);
      ct_tx(0x0100 + mode);
      break;
    }
  }
  delay(500);
  ct_rx_flush();			/* flush any other messages */
  if (mode == MODE_NORMAL)
    return(TRUE);
  ct_model = ct_get_model();		/* get type of phone */
  if (ct_model == 900)
  {
    ee_namindex = EE_NAMINDEX_9;
    ee_naminc   = EE_NAMINC_9;
    ee_namblock = EE_NAMBLOCK_9;
    ee_progpwd  = EE_PROGPWD_9;
    ee_custmsg  = EE_CUSTMSG_9;
    ee_namelen	= EE_NAMELEN_9;
    ram_strm_pos = RAM_STRM_POS_9;
    ram_strm_bit = RAM_STRM_BIT_9;
    ram_sysid    = RAM_SYSID_9;
  }
  else if (ct_model >= 1150)
  {
    ee_namindex = EE_NAMINDEX_11;
    ee_naminc   = EE_NAMINC_11;
    ee_namblock = EE_NAMBLOCK_11;
    ee_progpwd  = EE_PROGPWD_11;
    ee_custmsg  = EE_CUSTMSG_11;
    ee_namelen	= EE_NAMELEN_11;
    ram_strm_pos = RAM_STRM_POS_11;
    ram_strm_bit = RAM_STRM_BIT_11;
    ram_sysid    = RAM_SYSID_11;
  }
  ct_read_block(blk,ram_strm_pos,1);
  if (((blk[0] >> ram_strm_bit) & 0x01) == 0)
      ct_state.stream = 1;
    else
      ct_state.stream = 0;
  return(TRUE);
}

void ct_off()
{
  if (ct_model == 1200)
  {
    ct_pic_state(FALSE,FALSE);
    delay(750);
    ct_pic_state(TRUE,TRUE);
    delay(750);
  }
  ct_pic_state(FALSE,FALSE);
  delay(100);				/* let it get sent */
}

void ct_restore_state()
{
  ct_set_carrier(ct_state.carrier);
  ct_set_tx_power(ct_state.powerlevel);
  ct_set_signalling_tone(ct_state.sigtone);
  ct_set_rx_audio(ct_state.rxaudio);
  ct_get_rss();
  ct_set_tx_audio(ct_state.txaudio);
  ct_set_comp_exp(ct_state.compexp);
  ct_set_sat(ct_state.sat);
  ct_get_rss();
  ct_set_volume(ct_state.volume);
  ct_set_audio_path(ct_state.audiopath);
  ct_set_channel(ct_state.channel);
  ct_get_rss();
}

void ct_keypress(key)
  int key;
{
  word cmd;
  
  cmd = (0x0040 + key);
  ct_tx(cmd);
}

bint ct_get_bi()
{
  word cmd;
  byte a, b, c;
  
  ct_oki_send_data(3);
  ct_oki_get_msg(&cmd,&a,1000);
  ct_oki_get_msg(&cmd,&b,1000);
  ct_oki_get_msg(&cmd,&c,1000);
  return((c & 0x10) != 0);
}

void ct_set_carrier(state)
  bool state;
{
  if (state)
    ct_oki_send_data(7);
  else
    ct_oki_send_data(8);
  ct_state.carrier = state;
}

void ct_power(type,state)
  int type, state;
{
  if (type == PWR_TX)			/* tx circuits? */
    if (state)
      ct_oki_send_data(45);
    else
      ct_oki_send_data(46);
}

void ct_set_tx_power(level)
  int level;
{
  if ((level >=0) && (level <= 7))
  {
    ct_oki_send_data(10);
    ct_oki_send_data(level);
  }
  ct_state.powerlevel = level;
}

void ct_set_signalling_tone(state)
  bool state;
{
  if (state)
    ct_oki_send_data(16);
  else
    ct_oki_send_data(17);
  ct_state.sigtone = state;
}

void ct_set_audio_tone(tone,state)
  int tone;
  bool state;
{
  switch (tone)
  {
    case TONE_LOW:
       if (state) ct_oki_send_data(37); else ct_oki_send_data(38);
       break;
    case TONE_HIGH:
       if (state) ct_oki_send_data(35); else ct_oki_send_data(36);
       break;
    default:
       break;
  }
}

void ct_set_rx_audio(state)
  bool state;
{
  if (state)
    ct_oki_send_data(12);
  else
    ct_oki_send_data(11);
  ct_state.rxaudio = state;
}

void ct_set_tx_audio(state)
  bool state;
{
  if (state)
    ct_oki_send_data(14);
  else
    ct_oki_send_data(13);
  ct_state.txaudio = state;
}

void ct_set_comp_exp(state)
  bool state;
{
  if (state)
    ct_oki_send_data(65);
  else
    ct_oki_send_data(66);
  ct_state.compexp = state;
}

void ct_set_sat(sat)
  int sat;
{
  if (sat < 3)
  {
    ct_oki_send_data(32);
    ct_oki_send_data(sat);
  }
  else
  {
    ct_oki_send_data(33);		/* off */
  }
  ct_state.sat = sat;
}

void ct_set_volume(vol)
  int vol;
{
  ct_oki_send_data(67);
  ct_oki_send_data(vol);
  ct_state.volume = vol;
}

void ct_set_audio_path(path)
  int path;
{
  switch (path)
  {
    case AUDIO_EXTERNAL:  ct_oki_send_data(75);  break;
    case AUDIO_EARPIECE:  ct_oki_send_data(76);  break;
    case AUDIO_SOUNDER:   ct_oki_send_data(77);  break;
    default:  break;			 
  }
  ct_state.audiopath = path;
}

void ct_set_dtmf(tone)
  byte tone;
{
  char *p;
  int cmd;
  
  p = strchr(TX_TONE_CHARS,tone);
  if (p == NULL)
    return;
  cmd = TX_TONE_CHARS - p;
  if (cmd == 0)
  {
    ct_oki_send_data(43);		/* dtmf off */
  }
  else
  {
    ct_oki_send_data(42);
    ct_oki_send_data(cmd);
  }
}

int ct_dtmf_decode(command)
  int command;
{
  word msg;
  int i;
    
  switch (command)
  {
    case DTMF_DECODE_START:
      ct_oki_send_data(80);  return(TRUE);
    case DTMF_DECODE_GET:
      if (!ct_rx(&msg))	return(FALSE);
      i = msg & 0xff;
      if (i > 16) return(FALSE);
      return(TONE_CHARS[i]);
    case DTMF_DECODE_END:
      ct_oki_send_data(31); ct_rx_flush(); return(TRUE);
    default:  return(FALSE);
  }
}

int ct_fvc_msg(command,data)
  int command;
  byte *data;
{
  word msg;
  static byte rdata[6];
  static int rdp;
  byte bchword[6];
    
  switch (command)
  {
    case MSG_SETUP:
      ct_oki_send_data(21);
      rdp = 0;
      return(TRUE);
    case MSG_GET:
      for (;;)
      {
	if (!ct_rx(&msg)) return(FALSE);
	rdata[rdp] = msg & 0xff;
	if (++rdp >= 5)
	{
	  memcpy(bchword,rdata,5);
	  ct_gen_bch(28,bchword);	/* check bch */
	  if ((bchword[3] != rdata[3]) || (bchword[4] != rdata[4]))
	  {
	    ct_oki_send_data(21);
	    rdp = 0;
	    return(FALSE);
	  }
	  memcpy(data,rdata,5);
	  return(TRUE);
	}
      }
    case MSG_ABORT:
      ct_oki_send_data(56);
      delay(250);
      ct_rx_flush();
      return(TRUE);
    default:  return(FALSE);
  }
}

int ct_fcc_msg(command,data)
  int command;
  byte *data;
{
  word msg;
  static byte rdata[11];
  static int rdp;
  static time_t told;
  byte bchword[11];
    
  switch (command)
  {
    case MSG_SETUP:
      ct_oki_send_data(20);
      rdp = 0;
      return(TRUE);
    case MSG_GET:
      for (;;)
      {
	if (!ct_rx(&msg))
	{
	  if (rdp != 0)
	  {
	    if ((told + 1) < time(NULL))
	    {
	      ct_oki_send_data(56);
	      delay(250);
	      ct_rx_flush();
	      ct_oki_send_data(20);
	      rdp = 0;
	    }
	  }
	  return(FALSE);
	}
	time(&told);
	rdata[rdp] = msg & 0xff;
	if (++rdp >= 10)
	{
	  memcpy(bchword,rdata,5);
	  ct_gen_bch(28,bchword);	/* check bch, word1 */
	  if ((bchword[3] != rdata[3]) || (bchword[4] != rdata[4]))
	  {
	    ct_oki_send_data(20);
	    rdp = 0;
	    return(FALSE);
	  }
	  memcpy(bchword,rdata+5,5);
	  ct_gen_bch(28,bchword);	/* check bch, word2 */
	  if ((bchword[3] != rdata[8]) || (bchword[4] != rdata[9]))
	  {
	    ct_oki_send_data(20);
	    rdp = 0;
	    return(FALSE);
	  }
	  memcpy(data,rdata,10);
	  return(TRUE);
	}
      }
    case MSG_ABORT:
      ct_oki_send_data(56);
      delay(250);
      ct_rx_flush();
      return(TRUE);
    default:  return(FALSE);
  }
}

int ct_rcc_msg(dcc,data)
  int dcc;
  byte *data;
{
  int i;
  unsigned long chkesn;  

  if (ct_esn == 0)
    ct_get_esn(&ct_esn);
  chkesn = (((unsigned long) data[12] & 0x0f) << 28) +
	    (((unsigned long) data[13]) << 20) +
	    (((unsigned long) data[14]) << 12) +
	    (data[15] << 4)  +
	    ((data[16] & 0xf0) >> 4);
  if ((dcc < 0) || (dcc > 3)) return(FALSE);
  if (chkesn != ct_esn) return(FALSE);
  ct_oki_send_data(62);			/* transmit RCC message */
  switch (dcc)
  {
    case 0: ct_oki_send_data(0x00); break;
    case 1: ct_oki_send_data(0x1f); break;
    case 2: ct_oki_send_data(0x63); break;
    case 3: ct_oki_send_data(0x7c); break;
    default: return(FALSE);
  }
  for (i = 0; i < 30; i++)
    ct_oki_send_data(data[i]);
  return(TRUE);
}

int ct_rvc_msg(data)
  byte *data;
{
  int i;
  
  ct_oki_send_data(63);			/* transmit RVC message */
  for (i = 0; i < 12; i++)
    ct_oki_send_data(data[i]);
  return(TRUE);
}

void ct_set_channel(chnl)
  word chnl;
{
  ct_oki_send_data(9);
  ct_oki_send_data(chnl & 0xff);
  ct_oki_send_data((chnl >> 8) & 0xff);
  ct_state.channel = chnl;
}

int ct_get_rss()
{
  byte cmd, data;
  int rss;
  
  ct_oki_send_data(0x35);
  if (!ct_oki_get_msg(&cmd,&data,1000))
    return(-1);
  rss = data;
  if (ct_model >= 1150) rss += 3;
  if (rss < 64)
    rss = 0;
   else if (rss > 200)
    rss = -1;
   else rss -= 64;
  if (rss > 99)
    rss = 99;
  return(rss);
}

int ct_get_battery_level(volts)
  float *volts;
{
  byte cmd, data;
  byte v, last, c;

  last = 0;
  c = 0;
  if (ct_model == 900)
  {
    for (;;)
    {
      ct_oki_send_data(81);
      ct_oki_send_data(02);
      ct_oki_get_msg(&cmd,&data,1000);
      if (data == last) break;
      if (++c > 5) break;
    }
    if (volts != NULL)
      *volts = (float) ((data * 0.045) - 0.470);
  }
  if (ct_model >= 1150)
  {
    ct_get_rss();			/* cause a/d scan */
    ct_oki_send_data(25);
    ct_oki_send_data(0);
    ct_oki_send_data(0);
    ct_oki_send_data(0xd9);
    ct_oki_get_msg(&cmd,&data,1000);
    if (volts != NULL)
      *volts = (float) ((data * 0.0286) + 0.54);
    data = data - 45;
  }

  if (data >= 0x99) v=8;
   else if (data >= 0x97) v=7;
    else if (data >= 0x94) v=6;
     else if (data >= 0x91) v=5;
      else if (data >= 0x8f) v=4;
       else if (data >= 0x8c) v=3;
        else if (data >= 0x8a) v=2;
         else if (data >= 0x87) v=1;
          else v = 0;			       
  return(v);
}

int ct_get_sysid()
{
  byte sysid[4];
    
  if (!ct_read_block(sysid,ram_sysid,2))
    return(-1);
  return((int) (sysid[0] << 8) + sysid[1]);
}

int ct_read_mem(dest,addr,n)
  byte *dest;
  word addr;
  int n;  
{
  int i;
  byte cmd, data;
  
  for (i = 0; i < n; i++)
  {
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
    *dest++ = data;
    addr++;
  }
  *dest = '\0';
  return(TRUE);
}

int ct_read_block(dest,addr,n)
  byte *dest;
  word addr;
  int n;  
{
  int i;
  byte cmd, data;
  
  for (i = 0; i < n; i++)
  {
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
    *dest++ = data;
    ct_oki_incaddr(&addr);
  }
  *dest = '\0';
  return(TRUE);
}

int ct_set_block(src,addr,n)
  byte *src;
  word addr;
  int n;  
{
  int i;
  byte cmd, data;
  
  for (i = 0; i < n; i++)
  {
    ct_oki_send_data(54);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(src[i]);
    ct_get_rss();			/* delay till write is done */
    ct_oki_incaddr(&addr);
  }
  return(TRUE);
}

int ct_get_esn(esn)
  unsigned long *esn;
{
  byte cmd, data;
  byte a, b, c, d;

  if (ct_esn != 0)
    return(TRUE);
  ct_oki_send_data(24);
  if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
  a = data & 0xff;
  if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
  b = data & 0xff;
  if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
  c = data & 0xff;
  if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
  d = data & 0xff;
  *esn = (unsigned long) (((unsigned long) a) << 24) +
			 (((unsigned long) b) << 16) +
			 (((unsigned long) c) << 8)  +
			  ((unsigned long) d);
  ct_esn = *esn;
  return(TRUE);
}

int ct_get_model()
{
  byte cmd, data;
  byte a, b;

  *ct_rom_version = '\0';
  ct_oki_send_data(23);
  if (!ct_oki_get_msg(&cmd,&data,2000)) return(0);
  a = data;
  if (!ct_oki_get_msg(&cmd,&data,2000)) return(0);
  b = data;
  sprintf(ct_rom_version,"%02X%02X",a,b);
  if (!strcmp(ct_rom_version,"4003"))
    return(900);
  if (!strcmp(ct_rom_version,"4701"))
    return(900);
  if (!strcmp(ct_rom_version,"4904"))
    return(900); /* really 910 */
  if (!strcmp(ct_rom_version,"6025"))
    return(800);
  if (!strcmp(ct_rom_version,"6038"))
    return(1150);
  if (!strcmp(ct_rom_version,"6062"))
    return(1150);
  if (!strcmp(ct_rom_version,"6063"))
    return(1150);
  if (!strcmp(ct_rom_version,"6071"))
    return(1200);
  /* else try to figure out 900/1150 difference by number of bytes
     of address needed to read from memory */
  ct_oki_send_data(25);
  ct_oki_send_data(0);
  ct_oki_send_data(0);
  if (ct_oki_get_msg(&cmd,&data,500)) return(900);
  ct_oki_send_data(0);
  ct_oki_get_msg(&cmd,&data,500);
  return(1150);
}

char *ct_get_romver()
{
  return(ct_rom_version);
}

char *ct_get_prog_pwd()
{
  static char progpwd[12];
  byte ubuf[6];
    
  ct_read_block(ubuf,ee_progpwd,5);
  progpwd[0] = OKI_MEM_DEFS[((ubuf[0] >> 4) & 0x0f)];
  progpwd[1] = OKI_MEM_DEFS[(ubuf[0] & 0x0f)];
  progpwd[2] = OKI_MEM_DEFS[((ubuf[1] >> 4) & 0x0f)];
  progpwd[3] = OKI_MEM_DEFS[(ubuf[1] & 0x0f)];
  progpwd[4] = OKI_MEM_DEFS[((ubuf[2] >> 4) & 0x0f)];
  progpwd[5] = OKI_MEM_DEFS[(ubuf[2] & 0x0f)];
  progpwd[6] = OKI_MEM_DEFS[((ubuf[3] >> 4) & 0x0f)];
  progpwd[7] = OKI_MEM_DEFS[(ubuf[3] & 0x0f)];
  progpwd[8] = OKI_MEM_DEFS[((ubuf[4] >> 4) & 0x0f)];
  progpwd[9] = OKI_MEM_DEFS[(ubuf[4] & 0x0f)];
  progpwd[10] = '\0';
  return(progpwd);
}

char *ct_get_cust_msg()
{
  static char custmsg[12];
  
  ct_read_block(custmsg,ee_custmsg,ee_namelen);
  custmsg[ee_namelen] = '\0';
  return(custmsg);
}

int ct_set_cust_msg(msg)
  byte *msg;
{
  if (strlen(msg) > ee_namelen)
    return(FALSE);  
  ct_set_block(msg,ee_custmsg,ee_namelen);
  return(TRUE);
}

/* read 67 byte checksummed nam info block */
int ct_get_namblock()
{
  int i, sum;
  byte cmd, data;
  word addr;
  
  addr = ee_namblock;
  for (i = 0; i < 67; i++)
  {
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    addr = addr + ee_naminc;
    ct_oki_send_data(25);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    addr = addr + ee_naminc;
    if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
    ct_namblock[i] = data;
    i++;
    if (!ct_oki_get_msg(&cmd,&data,1000)) return(FALSE);
    ct_namblock[i] = data;
  }
  /* test for correct checksum */
  sum = 0;
  for (i = 0; i < 67; i++)
    sum = sum + ct_namblock[i];
  sum = sum & 0xff;
  if (sum == 0) ct_got_namblock = TRUE;
  return(sum == 0);
}

int ct_get_nams()
{
  int i, ac, pre, mid, end, sid;
  
  if (!ct_get_namblock())
    return(FALSE);
  for (i = 0; i < 5; i++)
  {
    ac  = (ct_namblock[i*11+2] << 4) + (ct_namblock[i*11+3] >> 4);
    pre = ((ct_namblock[i*11+3] & 0x0f) << 6) + (ct_namblock[i*11+4] >> 2);
    mid = ((ct_namblock[i*11+4] & 0x03) << 2) + (ct_namblock[i*11+5] >> 6);
    if (mid == 10) mid = 0;
    end = ((ct_namblock[i*11+5] & 0x3f) << 4) + (ct_namblock[i*11+6] >> 4);
    sprintf(nam_info[i].number,"(%03d) %03d-%d%03d",
                ct_minunpack(ac),ct_minunpack(pre),mid,ct_minunpack(end));
    sid = (ct_namblock[i*11] << 8) + ct_namblock[i*11+1];
    sprintf(nam_info[i].sid,"%05d",sid);
    if (sid & 0x0001) 
      nam_info[i].sys = 'A';
    else
      nam_info[i].sys = 'B';
    sprintf(nam_info[i].ipch,"%04d",
                             (ct_namblock[i*11+7] << 8) + ct_namblock[i*11+8]);
    sprintf(nam_info[i].olc,"%02d",ct_namblock[i*11+9]);
    sprintf(nam_info[i].gim,"%02d",ct_namblock[i*11+10]);
  }
  return(TRUE);
}

/*
  set nams according to ascii info in struct nam_info.
  if invalid info or write fails, return FALSE.
*/
int ct_set_nams()
{
  char s[8];
  char *p;
  int n, nam;
  
  if (!ct_got_namblock) return(FALSE);
  for (nam = 0; nam < 5; nam++)
  {
    if (nam_info[nam].number[0] != '(') return(FALSE);
    if (nam_info[nam].number[4] != ')') return(FALSE);
    if (nam_info[nam].number[5] != ' ') return(FALSE);
    if (nam_info[nam].number[9] != '-') return(FALSE);
    s[0] = nam_info[nam].number[1];
    s[1] = nam_info[nam].number[2];
    s[2] = nam_info[nam].number[3];
    s[3] = '\0';
    n = ct_minpack(s);
    ct_namblock[nam*11+2] = n >> 4;
    ct_namblock[nam*11+3] = (n & 0x0f) << 4;

    s[0] = nam_info[nam].number[6];
    s[1] = nam_info[nam].number[7];
    s[2] = nam_info[nam].number[8];
    s[3] = '\0';
    n = ct_minpack(s);
    ct_namblock[nam*11+3] += (n >> 6);
    ct_namblock[nam*11+4] =  (n << 2) & 0xff;

    n = nam_info[nam].number[10] - '0';
    if (n == 0) n = 10;
    ct_namblock[nam*11+4] += (n >> 2);
    ct_namblock[nam*11+5] =  (n & 0x03) << 6;

    s[0] = nam_info[nam].number[11];
    s[1] = nam_info[nam].number[12];
    s[2] = nam_info[nam].number[13];
    s[3] = '\0';
    n = ct_minpack(s);
    ct_namblock[nam*11+5] += (n >> 4);
    ct_namblock[nam*11+6] =  (n & 0x0f) << 4;

    n = atoi(nam_info[nam].sid);
    ct_namblock[nam*11]   = (n >> 8);
    ct_namblock[nam*11+1] = (n & 0xff);

    n = atoi(nam_info[nam].ipch);
    ct_namblock[nam*11+7] = (n >> 8);
    ct_namblock[nam*11+8] = (n & 0xff);

    n = atoi(nam_info[nam].olc);
    ct_namblock[nam*11+9] = n;

    n = atoi(nam_info[nam].gim);
    ct_namblock[nam*11+10] = n;
  }
  return(ct_set_namblock());
}

/*
  send ct_namblock array to phone.  returns TRUE if ok.
  generate proper checksum first, and verify and sending.
*/
int ct_set_namblock()
{
  int i, sum;
  word addr;
  
  /* gen checksum */
  sum = 0x55;
  for (i = 0; i < 65; i++)
    sum = sum + ct_namblock[i];
  sum = (~(sum & 0xff)) + 1;
  ct_namblock[65] = sum;
  ct_namblock[66] = 0x55;

  /* check the checksum! */
  sum = 0;
  for (i = 0; i < 67; i++)
    sum = sum + ct_namblock[i];
  sum = sum & 0xff;
  if (sum != 0)
    return(FALSE);

  if (ct_model == 900)
  {
    /* enable writes to NAM area */
    addr = 0x7005;
    ct_oki_send_data(54);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(1);
    ct_get_rss();
  }
  
  /* now save to phone */
  addr = ee_namblock;
  for (i = 0; i < 67; i++)
  {
    ct_oki_send_data(54);
    if (ct_model >= 1150)
      ct_oki_send_data(0);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(ct_namblock[i]);
    if ((i % 2) == 0)
      ct_get_rss();			/* delay till write is done */
    addr = addr + ee_naminc;
  }

  if (ct_model == 900)
  {
    /* disable writes to NAM area */
    addr = 0x7005;
    ct_oki_send_data(54);
    ct_oki_send_data((addr >> 8) & 0xff);
    ct_oki_send_data( addr       & 0xff);
    ct_oki_send_data(0);
  }
  ct_get_rss();
  return(TRUE);
}


int ct_stream(stream)
  int stream;
{
  byte blk[8];
  byte bit;
      
  if (stream == -1)			/* read stream */
    return(ct_state.stream);
  if (stream == ct_state.stream)	/* duh */
    return(stream);
  /* change stream */
  delay(250);
  ct_rx_flush();
  if (stream == 0) bit = 1;
    else bit = 0;
  ct_read_block(blk,ram_strm_pos,1);
  blk[0] = (blk[0] & ~(1 << ram_strm_bit)) | (bit << ram_strm_bit);
  ct_set_block(blk,ram_strm_pos,1);
  ct_state.stream = stream;

  return(ct_state.stream);  
}


/************** utility routines that don't control phone *************/

/* increment channel to next channel, or wrap */
word ct_inc_channel(channel)
  word channel;
{
  switch (++channel)
  {
    case 313:  channel = 667; break;
    case 717:  channel = 991; break;
    case 1024: channel = 1;   break;
    case 667:  channel = 717; break;
    case 800:  channel = 355; break;
    default:   break;
  }
  return(channel);
}

/* decrement channel to next channel, or wrap */
word ct_dec_channel(channel)
  word channel;
{
  switch (--channel)
  {
    case 0:    channel = 1023; break;
    case 716:  channel = 666;  break;
    case 354:  channel = 799;  break;
    case 990:  channel = 716;  break;
    case 666:  channel = 312;  break;
    default:   break;
  }
  return(channel);
}

bint ct_channel_type(channel,system,type)
  word channel;
  byte *system;
  byte *type;
{
  if ((channel >= 1) && (channel <= 312))
  {
    *system = 'A';
    *type = 'A';
    return(TRUE);
  }
  if ((channel >= 313) && (channel <= 333))
  {
    *system = 'A';
    *type = 'C';
    return(TRUE);
  }
  if ((channel >= 334) && (channel <= 354))
  {
    *system = 'B';
    *type = 'C';
    return(TRUE);
  }
  if ((channel >= 355) && (channel <= 666))
  {
    *system = 'B';
    *type = 'A';
    return(TRUE);
  }
  if ((channel >= 667) && (channel <= 716))
  {
    *system = 'A';
    *type = 'A';
    return(TRUE);
  }
  if ((channel >= 717) && (channel <= 799))
  {
    *system = 'B';
    *type = 'A';
    return(TRUE);
  }
  if ((channel >= 991) && (channel <= 1023))
  {
    *system = 'A';
    *type = 'A';
    return(TRUE);
  }
  *system = 'N';
  *type = 'N';
  return(FALSE);
}

int ct_decode_fvc_msg(msg,scc,pscc,order,ordq,local,chan,vmac)
  byte *msg;
  int *scc,*pscc,*order,*ordq,*local,*chan,*vmac;
{
  if ((msg[0] & 0xc0) != 0x80)
    return(FALSE);
  *scc   = (msg[0] & 0x30) >> 4;
  *pscc  = (msg[0] & 0x0c) >> 2;
  *order = ((msg[2] & 0x01) << 4) + (msg[3] >> 4);
  *ordq  = ((msg[2] & 0x0e) >> 1);
  *local = ((msg[1] & 0x01) << 4) + ((msg[2] & 0xf0) >> 4);
  *chan  = ((msg[2] & 0x7f) << 4) + ((msg[3] & 0xf0) >> 4);
  *vmac  = ((msg[1] & 0x03) << 1) + ((msg[2] & 0x80) >> 7);
  if ((*scc != 3) && ((*chan < 1) || (*chan > 1023)))
    return(FALSE);
  return(TRUE);
}

int ct_decode_fcc_msg(msg,scc,dcc,min,order,ordq,local,chan,vmac)
  byte *msg;
  byte **min;
  int *scc,*dcc,*order,*ordq,*local,*chan,*vmac;
{
  static char telnum[16];  /* (nnn) nnn-nnnn */
  int min1p, min1c, min1n, min2;
  
  if ((msg[0] & 0xc0) != 0x40)
    return(FALSE);
  if ((msg[5] & 0xc0) != 0x80)
    return(FALSE);
  *dcc   = (msg[0] & 0x30) >> 4;
  *scc   = (msg[5] & 0x30) >> 4;
  *order = ((msg[7] & 0x01) << 4) + (msg[8] >> 4);
  *ordq  = ((msg[7] & 0x0e) >> 1);
  *local = ((msg[6] & 0x01) << 4) + ((msg[7] & 0xf0) >> 4);
  *vmac  = ((msg[6] & 0x03) << 1) + ((msg[7] & 0x80) >> 7);
  *chan  = ((msg[7] & 0x7f) << 4) + ((msg[8] & 0xf0) >> 4);
  if ((*scc != 3) && ((*chan < 1) || (*chan > 1023)))
    return(FALSE);
  min1p  = ((msg[0] & 0x0f) << 6) + ((msg[1] & 0xfc) >> 2);
  min1c  = ((msg[1] & 0x03) << 2) + ((msg[2] & 0xc0) >> 6);
  if (min1c == 10) min1c = 0;
  min1n  = ((msg[2] & 0x3f) << 4) + ((msg[3] & 0xf0) >> 4);
  min2   = ((msg[5] & 0x0f) << 6) + ((msg[6] & 0xfc) >> 2);
  sprintf(telnum,"(%03d) %03d-%d%03d",ct_minunpack(min2),
	  ct_minunpack(min1p),min1c,ct_minunpack(min1n));
  *min = telnum;
  return(TRUE);
}

char *ct_decode_order(order,ordq)
  int order, ordq;
{
  static char ords[64];
  
  switch (order)
  {
    case  0:  return("page");
    case  1:  return("alert");
    case  3:  return("release");
    case  4:  return("reorder");
    case  6:  return("stop alert");
    case  7:  return("audit");
    case  8:  return("send called-address");
    case  9:  return("intercept");
    case 10:  return("maintenance");
    case 11:  sprintf(ords,"set power level %d",ordq); return(ords);
    case 12:  if (ordq == 0)
		return("directed retry - not last"); 
	      else if (ordq == 1)
		return("directed retry - last");
	      break;
    case 13:  if (ordq == 0)
	        return("non-auton reg: off");
	      else if (ordq == 1)
	        return("non-auton reg: on");
	      else if (ordq == 2)
	        return("auton reg: off");
	      else if (ordq == 3)
	        return("auton reg: on");
	      break;
    case 30:  return("local control");
    default:  break;
  }
  sprintf(ords,"reserved: %d,%d",order,ordq);
  return(ords);
}

/******************* Internal routines below this line ***********************/

/* send PIC commands to set oki state lines */
ct_pic_state(powerp,portablep)
  bool powerp;
  bool portablep;
{
/* portable line no longer connected to phone in okifrob */
  tty_tx(0x80+'@');
  tty_tx(0x80+'@');
  tty_tx(0xa2+portablep);
  
  tty_tx(0x80+'@');
  tty_tx(0x80+'@');
  tty_tx(0xa0+powerp);
}

/*
   get a response from the oki and return true and cmd and data.
   if no response after timeout milliseconds return false 
*/
bint ct_oki_get_msg(cmd,data,timeout)
  byte *cmd;
  byte *data;
  word timeout;
{
  word i, msg;
  
  for (i = 0; i < timeout; i++)
  {
    if (ct_rx(&msg)) break;
    delay(1);
  }
  if (i == timeout)
  {
    *cmd = 0;
    *data = 0;
    return(FALSE);
  }
  *cmd  = msg >> 8;
  *data = msg & 0x00ff;
  return(TRUE);
}

/* send the passed data byte to the oki */
ct_oki_send_data(data)
  byte data;
{
  word cmd;
  
  cmd = (0x0700 + data);
  ct_tx(cmd);
}

/* put 16-bit command code in transmit queue to oki */
ct_tx(msg)
  word msg;
{
  byte hexs[4];
  
  hexs[0] = 0xc0 + ((msg & 0x0400) >> 10);
  hexs[1] = 0xc0 + ((msg & 0x03f0) >> 4);
  hexs[2] = 0xb0 +  (msg & 0x000f);
  hexs[3] = '\0';
  tty_txs(hexs);
/* printf("[%04x]",msg); */
}

void ct_rx_flush()
{
  word msg;

  while (ct_rx(&msg));  
}

bint ct_rx(msg)
  word *msg;
{
  byte c;
  
  while (tty_rx(&c))
  {
    rxbuf[0] = rxbuf[1];
    rxbuf[1] = rxbuf[2];
    rxbuf[2] = c;
    if ((c & 0xf0) == 0xb0)
    {
      *msg = ((rxbuf[0] & 0x3f) << 10) |
	     ((rxbuf[1] & 0x3f) << 4)  |
	      (rxbuf[2] & 0x0f);
/* printf("{%04x}",*msg); */
      return(TRUE);
    }
  }
  return(FALSE);
}

ct_oki_incaddr(addr)
  word *addr;
{
  for (;;)
  {
    (*addr)++;
    if (ct_model >= 1150)
      break;
    if ((*addr & 0x003e) != 0x002a)
      break;
  }
}


/*============================BCH STUFF=========================*/

/*
 * 48 bit word:
 * BCH(48, 36; 5)
 *
 * 40 bit word:
 * BCH(40, 28; 5)
 *
 * 1 + x^3 + x^4 + x^5 + x^8 + x^10 + x^12
 * TS
 */

#define	BCH_ORDER	12
#define	BCH_GEN (BCH_X(0)+BCH_X(3)+BCH_X(4)+BCH_X(5)+BCH_X(8)+BCH_X(10)+BCH_X(12))

/*
 * Evaluate BCH(n, k; d), (n-k) < 32
 */

#define BCH_X(x)	(1<<(x))
#define BCH_MASK	(BCH_X(BCH_ORDER)-1)

unsigned int ct_sr;

ct_bchinit()
{
	ct_sr = 0;
}

ct_bchbit(x)
unsigned int x;
{
	unsigned int s = ct_sr;

	s = (s << 1);
	if(x ^ !!(s & (BCH_X(BCH_ORDER))))
		s ^= BCH_GEN;
	return(ct_sr = s);
}

unsigned int ct_bchcode()
{
	return(ct_sr & (BCH_MASK));
}

/* takes number of bits to generate a bch code for,
   and pointer to first byte of data buffer.
   deposits 12-bit bch into buffer right after the data bits */
unsigned int ct_gen_bch(bits,buf)
  int bits;
  unsigned char *buf;
{
  int i, b, bch;
  unsigned char *p;

  p = buf;  
  b = 8;
  ct_bchinit();
  for (i = 0; i < bits; i++)
  {
    b--;
    ct_bchbit(((*p >> b) & 0x01));
    if (b == 0)
    {
      b = 8;
      p++;
    }
  }
  bch = ct_bchcode();
  for (i = 11; i >= 0; i--)
  {
    b--;
    *p = (byte) (*p & ~(1 << b));
    *p = (byte) (*p | (((bch >> i) & 0x01) << b));
    if (b == 0)
    {
      b = 8;
      p++;
    }
  }
}


/*============================MIN STUFF=========================*/
int ct_minunpack(i)
  int i;
{
  long d;

  d = ((i + 100) % 1000) / 100;
  d = d*10 + ((i + 10) % 100) / 10;
  d = d*10 + ((i + 1) % 10);
  return((int) d);
}

int ct_minpack(s)
  char *s;
{
  int i, d;
  
  d = s[0] - '0';
  if (d == 0) d = 10;
  i = d * 100;
  d = s[1] - '0';
  if (d == 0) d = 10;
  i = i + (d * 10);
  d = s[2] - '0';
  if (d == 0) d = 10;
  i = i + d;
  i = i - 111;
  return(i);
}

/* takes phone number in (nnn) nnn-nnnn format, return min1 and min2 */
void ct_tel2min(telnum,min1,min2)
  char *telnum;
  unsigned long *min1, *min2;
{
  int d;
  unsigned long tmp;

  *min2 = ct_minpack(telnum+1);
  tmp =   ((unsigned long) ct_minpack(telnum+6)) << 14;
  d =     telnum[10] - '0'; if (d == 0) d = 10;
  tmp =   tmp | (d << 10);
  *min1 = tmp | ct_minpack(telnum+11);
}
